<?php


// namespace App\Exports;

// use Illuminate\Contracts\View\View;
// use Maatwebsite\Excel\Concerns\FromView;
// use Maatwebsite\Excel\Concerns\WithStyles;
// use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

// class ProductExport implements FromView, WithStyles
// {
//     protected $products;
//     protected $product_attribute;
//     public $rowStyles = [];

//     public function __construct($products, $product_attribute)
//     {
//         $this->products = $products;
//         $this->product_attribute = $product_attribute;

//         $this->prepareRowStyles(); // 🟡 Move logic here
//     }

//     private function prepareRowStyles()
//     {
//         $rowNum = 2; // Header is at row 1

//         foreach ($this->products as $prod) {
//             if ($prod->has_attribute === 'Y') {
//                 $this->rowStyles[] = $rowNum; // Mark row for yellow background
//             }

//             $rowNum++; // Main product row

//             // Account for attribute rows
//             if (
//                 $prod->has_attribute === 'Y'
//                 && isset($this->product_attribute[$prod->id])
//                 && is_iterable($this->product_attribute[$prod->id])
//             ) {
//                 foreach ($this->product_attribute[$prod->id] as $attr) {
//                     $rowNum++; // Each attribute row
//                 }
//             }
//         }
//     }

//     public function view(): View
//     {
//         return view('admin.product.excel', [
//             'product' => $this->products,
//             'product_attribute' => $this->product_attribute
//         ]);
//     }

//     public function styles(Worksheet $sheet)
//     {
//         $styles = [];

//         // Header styling
//         $styles['A1:N1'] = [
//             'font' => ['bold' => true],
//             'fill' => [
//                 'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
//                 'startColor' => ['rgb' => 'DDDDDD'],
//             ],
//         ];

//         // Highlight parent rows (has_attribute = 'Y')
//         foreach ($this->rowStyles as $rowNum) {
//             $styles["A{$rowNum}:N{$rowNum}"] = [
//                 'fill' => [
//                     'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
//                     'startColor' => ['rgb' => 'FFFF99'], // Light Yellow
//                 ],
//             ];
//         }

//         return $styles;
//     }
// }



namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithHeadings;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use App\Http\Helper1;

class ProductExport implements FromCollection, WithStyles, WithHeadings
{
    protected $products;
    protected $productAttributes;
    protected $rowStyles = [];

    public function __construct($products, $productAttributes)
    {
        $this->products = $products;
        $this->productAttributes = $productAttributes;
    }

    public function headings(): array
    {
        return [
            'Product ID',
            'Product Name',
            'Product Link',
            'Description',
            'Content',
            'Price',
            'Actual Price',
            'Block Status',
            'Delay Days',
            'Delivery Type',
            'Specification',
            'Segregation',
            'Vendor Name',
            'Category',
        ];
    }

    public function collection()
    {
        $collection = collect();
        $rowNum = 2;

        foreach ($this->products as $prod) {
            if (!is_object($prod)) continue;

            if ($prod->has_attribute === 'Y') {
                $this->rowStyles[] = $rowNum;
            }

            $categoriesCollection = Helper1::product_relatated_category_list($prod->id);

            if ($categoriesCollection instanceof \Illuminate\Support\Collection) {
                $categories = implode(' | ', $categoriesCollection->toArray());
            } else {
                $categories = implode(' | ', (array)$categoriesCollection);
            }
            $collection->push([
                $prod->fnid,
                $prod->product_name,
                $prod->slug,
                //htmlspecialchars($prod->description),
                //htmlspecialchars($prod->content),
                $this->cleanText($prod->description),
                //$this->cleanText($prod->content),
                $this->escapeExcelFormula($prod->content),
                $prod->price,
                $prod->actual_price,
                $prod->is_block,
                $prod->delivery_delay_days,
                $prod->delivery_type,
                $prod->specification,
                $prod->segregation,
                $prod->vendor_name,
                $categories,
            ]);

            $rowNum++;

            if ($prod->has_attribute === 'Y' &&isset($this->productAttributes[$prod->id]) &&is_iterable($this->productAttributes[$prod->id])
            ) {
                foreach ($this->productAttributes[$prod->id] as $attr) {
                    $collection->push([
                        $attr->attribute_fnid,
                        $prod->product_name,
                        $prod->slug,
                        $this->cleanText($prod->description),
                        //$this->cleanText($prod->content),
                        $this->escapeExcelFormula($prod->content),
                        $attr->price,
                        $attr->actual_price,
                        $attr->is_block,
                        $prod->delivery_delay_days,
                        $prod->delivery_type,
                        $prod->specification,
                        $prod->segregation,
                        $prod->vendor_name,
                        $categories,
                    ]);
                    $rowNum++;
                }
            }
        }

        return $collection;
    }

    private function cleanText($text)
    {
        return str_replace(
            ['â€¢', '<br>', '<br/>', '<br />'],
            ['•', "\n", "\n", "\n"],
            html_entity_decode($text)
        );
    }

    private function escapeExcelFormula($text)
    {
        $clean = $this->cleanText($text);

        // If content starts with characters Excel interprets as formula triggers (=, +, -, @)
        // if (preg_match('/^[=+\-@]/', ltrim($clean))) {
        //     return "'" . $clean; // Prefix with apostrophe to force Excel to treat as text
        // }

        $trimmed = ltrim($clean);

        // Only escape if starts with '=', '+', '-', or '@'
        if (preg_match('/^[=+\-@]/', $trimmed)) {
            return "'" . $clean;
        }


        return $clean;
    }

    public function styles(Worksheet $sheet)
    {
        $styles = [];

        // Header row style
        $styles['A1:N1'] = [
            'font' => ['bold' => true],
            'fill' => [
                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                'startColor' => ['rgb' => 'FFFF99'],
            ],
        ];

        // Yellow background for parent rows
        foreach ($this->rowStyles as $rowNum) {
            $styles["A{$rowNum}:N{$rowNum}"] = [
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'FFFF99'], // Light Yellow
                ],
            ];
        }

        $highestRow = $sheet->getHighestRow();
        for ($row = 2; $row <= $highestRow; $row++) {
            $sheet->getStyle("D{$row}")->getAlignment()->setWrapText(true); // Description
            $sheet->getStyle("E{$row}")->getAlignment()->setWrapText(true); // Content
        }



        return $styles;
    }
}
